"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.wrapWithKeyObjectIfSupported = exports.unwrapDataKey = exports.subtleFunctionForMaterial = exports.keyUsageForMaterial = exports.isValidCryptoKey = exports.isCryptoKey = exports.decorateWebCryptoMaterial = exports.decorateDecryptionMaterial = exports.decorateEncryptionMaterial = exports.decorateCryptographicMaterial = exports.isBranchKeyMaterial = exports.isDecryptionMaterial = exports.isEncryptionMaterial = exports.WebCryptoDecryptionMaterial = exports.WebCryptoEncryptionMaterial = exports.NodeDecryptionMaterial = exports.NodeEncryptionMaterial = exports.NodeBranchKeyMaterial = exports.supportsKeyObject = void 0;
const encrypted_data_key_1 = require("./encrypted_data_key");
const signature_key_1 = require("./signature_key");
const immutable_class_1 = require("./immutable_class");
const keyring_trace_1 = require("./keyring_trace");
const node_algorithms_1 = require("./node_algorithms");
const web_crypto_algorithms_1 = require("./web_crypto_algorithms");
const needs_1 = require("./needs");
const uuid_1 = require("uuid");
exports.supportsKeyObject = (function () {
    try {
        const { KeyObject, createSecretKey } = require('crypto'); // eslint-disable-line @typescript-eslint/no-var-requires
        if (!KeyObject || !createSecretKey)
            return false;
        return { KeyObject, createSecretKey };
    }
    catch (ex) {
        return false;
    }
})();
/*
 * This public interface to the CryptographicMaterial object is provided for
 * developers of CMMs and keyrings only. If you are a user of the AWS Encryption
 * SDK and you are not developing your own CMMs and/or keyrings, you do not
 * need to use it and you should not do so.
 *
 * The CryptographicMaterial's purpose is to bind together all the required elements for
 * encrypting or decrypting a payload.
 * The functional data key (unencrypted or CryptoKey) is the most sensitive data and needs to
 * be protected.  The longer this data persists in memory the
 * greater the opportunity to be invalidated.  Because
 * a Caching CMM exists it is important to ensure that the
 * unencrypted data key and its meta data can not be manipulated,
 * and that the unencrypted data key can be zeroed when
 * it is no longer needed.
 */
const timingSafeEqual = (function () {
    try {
        /* It is possible for `require` to return an empty object, or an object
         * that does not implement `timingSafeEqual`.
         * in this case I need a fallback
         */
        const { timingSafeEqual: nodeTimingSafeEqual } = require('crypto'); // eslint-disable-line @typescript-eslint/no-var-requires
        return nodeTimingSafeEqual || portableTimingSafeEqual;
    }
    catch (e) {
        return portableTimingSafeEqual;
    }
    /* https://codahale.com/a-lesson-in-timing-attacks/ */
    function portableTimingSafeEqual(a, b) {
        /* It is *possible* that a runtime could optimize this constant time function.
         * Adding `eval` could prevent the optimization, but this is no guarantee.
         * The eval below is commented out
         * because if a browser is using a Content Security Policy with `'unsafe-eval'`
         * it would fail on this eval.
         * The value in attempting to ensure that this function is not optimized
         * is not worth the cost of making customers allow `'unsafe-eval'`.
         * If you want to copy this function for your own use,
         * please review the timing-attack link above.
         * Side channel attacks are pernicious and subtle.
         */
        // eval('') // eslint-disable-line no-eval
        /* Check for early return (Postcondition) UNTESTED: Size is well-know information
         * and does not leak information about contents.
         */
        if (a.byteLength !== b.byteLength)
            return false;
        let diff = 0;
        for (let i = 0; i < b.length; i++) {
            diff |= a[i] ^ b[i];
        }
        return diff === 0;
    }
})();
class NodeBranchKeyMaterial {
    // all attributes are readonly so they are accessible outside the class but
    // they cannot be modified
    // since all fields are objects, keep them immutable from external changes via
    // shared memory
    // we want the branch key to be mutable within the class but immutable outside
    // the class
    _branchKey;
    constructor(branchKey, branchKeyIdentifier, branchKeyVersion, encryptionContext) {
        /* Precondition: Branch key must be a Buffer */
        (0, needs_1.needs)(branchKey instanceof Buffer, 'Branch key must be a Buffer');
        /* Precondition: Branch key id must be a string */
        (0, needs_1.needs)(typeof branchKeyIdentifier === 'string', 'Branch key id must be a string');
        /* Precondition: Branch key version must be a string */
        (0, needs_1.needs)(typeof branchKeyVersion === 'string', 'Branch key version must be a string');
        /* Precondition: encryptionContext must be an object, even if it is empty */
        (0, needs_1.needs)(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be an object');
        /* Precondition: branchKey must be a 32 byte-long buffer */
        (0, needs_1.needs)(branchKey.length === 32, 'Branch key must be 32 bytes long');
        /* Precondition: branch key ID is required */
        (0, needs_1.needs)(branchKeyIdentifier, 'Empty branch key ID');
        /* Precondition: branch key version must be valid version 4 uuid */
        (0, needs_1.needs)((0, uuid_1.validate)(branchKeyVersion) && (0, uuid_1.version)(branchKeyVersion) === 4, 'Branch key version must be valid version 4 uuid');
        /* Postcondition: branch key is immutable */
        this._branchKey = Buffer.from(branchKey);
        /* Postconditon: encryption context is immutable */
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        this.branchKeyIdentifier = branchKeyIdentifier;
        this.branchKeyVersion = Buffer.from(branchKeyVersion, 'utf-8');
        Object.setPrototypeOf(this, NodeBranchKeyMaterial.prototype);
        /* Postcondition: instance is frozen */
        // preventing any modifications to its properties or methods.
        Object.freeze(this);
    }
    // makes the branch key public to users wrapped in immutable access
    branchKey() {
        return this._branchKey;
    }
    // this capability is not required of branch key materials according to the
    // specification. Using this is a good security practice so that the data
    // key's bytes are not preserved even in free memory
    zeroUnencryptedDataKey() {
        this._branchKey.fill(0);
        return this;
    }
}
exports.NodeBranchKeyMaterial = NodeBranchKeyMaterial;
// make the class immutable
(0, immutable_class_1.frozenClass)(NodeBranchKeyMaterial);
class NodeEncryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptedDataKeys;
    addEncryptedDataKey;
    setSignatureKey;
    signatureKey;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: NodeEncryptionMaterial suite must be NodeAlgorithmSuite. */
        (0, needs_1.needs)(suite instanceof node_algorithms_1.NodeAlgorithmSuite, 'Suite must be a NodeAlgorithmSuite');
        this.suite = suite;
        /* Precondition: NodeEncryptionMaterial encryptionContext must be an object, even if it is empty. */
        (0, needs_1.needs)(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // EncryptionMaterial have generated a data key on setUnencryptedDataKey
        const setFlags = keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        decorateEncryptionMaterial(this);
        Object.setPrototypeOf(this, NodeEncryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
exports.NodeEncryptionMaterial = NodeEncryptionMaterial;
(0, immutable_class_1.frozenClass)(NodeEncryptionMaterial);
class NodeDecryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setVerificationKey;
    verificationKey;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: NodeDecryptionMaterial suite must be NodeAlgorithmSuite. */
        (0, needs_1.needs)(suite instanceof node_algorithms_1.NodeAlgorithmSuite, 'Suite must be a NodeAlgorithmSuite');
        this.suite = suite;
        /* Precondition: NodeDecryptionMaterial encryptionContext must be an object, even if it is empty. */
        (0, needs_1.needs)(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // DecryptionMaterial have decrypted a data key on setUnencryptedDataKey
        const setFlags = keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        decorateDecryptionMaterial(this);
        Object.setPrototypeOf(this, NodeDecryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
exports.NodeDecryptionMaterial = NodeDecryptionMaterial;
(0, immutable_class_1.frozenClass)(NodeDecryptionMaterial);
class WebCryptoEncryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptedDataKeys;
    addEncryptedDataKey;
    setSignatureKey;
    signatureKey;
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: WebCryptoEncryptionMaterial suite must be WebCryptoAlgorithmSuite. */
        (0, needs_1.needs)(suite instanceof web_crypto_algorithms_1.WebCryptoAlgorithmSuite, 'Suite must be a WebCryptoAlgorithmSuite');
        this.suite = suite;
        this.validUsages = Object.freeze([
            'deriveKey',
            'encrypt',
        ]);
        /* Precondition: WebCryptoEncryptionMaterial encryptionContext must be an object, even if it is empty. */
        (0, needs_1.needs)(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // EncryptionMaterial have generated a data key on setUnencryptedDataKey
        const setFlag = keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateEncryptionMaterial(this);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoEncryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey && this.hasCryptoKey;
    }
}
exports.WebCryptoEncryptionMaterial = WebCryptoEncryptionMaterial;
(0, immutable_class_1.frozenClass)(WebCryptoEncryptionMaterial);
class WebCryptoDecryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setVerificationKey;
    verificationKey;
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: WebCryptoDecryptionMaterial suite must be WebCryptoAlgorithmSuite. */
        (0, needs_1.needs)(suite instanceof web_crypto_algorithms_1.WebCryptoAlgorithmSuite, 'Suite must be a WebCryptoAlgorithmSuite');
        this.suite = suite;
        this.validUsages = Object.freeze([
            'deriveKey',
            'decrypt',
        ]);
        /* Precondition: WebCryptoDecryptionMaterial encryptionContext must be an object, even if it is empty. */
        (0, needs_1.needs)(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // DecryptionMaterial have decrypted a data key on setUnencryptedDataKey
        const setFlag = keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateDecryptionMaterial(this);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoDecryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasCryptoKey;
    }
}
exports.WebCryptoDecryptionMaterial = WebCryptoDecryptionMaterial;
(0, immutable_class_1.frozenClass)(WebCryptoDecryptionMaterial);
function isEncryptionMaterial(obj) {
    return (obj instanceof WebCryptoEncryptionMaterial ||
        obj instanceof NodeEncryptionMaterial);
}
exports.isEncryptionMaterial = isEncryptionMaterial;
function isDecryptionMaterial(obj) {
    return (obj instanceof WebCryptoDecryptionMaterial ||
        obj instanceof NodeDecryptionMaterial);
}
exports.isDecryptionMaterial = isDecryptionMaterial;
function isBranchKeyMaterial(obj) {
    return obj instanceof NodeBranchKeyMaterial;
}
exports.isBranchKeyMaterial = isBranchKeyMaterial;
function decorateCryptographicMaterial(material, setFlag) {
    /* Precondition: setFlag must be in the set of KeyringTraceFlag.SET_FLAGS. */
    (0, needs_1.needs)(setFlag & keyring_trace_1.KeyringTraceFlag.SET_FLAGS, 'Invalid setFlag');
    /* When a KeyringTraceFlag is passed to setUnencryptedDataKey,
     * it must be valid for the type of material.
     * It is invalid to claim that EncryptionMaterial were decrypted.
     */
    const deniedSetFlags = (keyring_trace_1.KeyringTraceFlag.SET_FLAGS ^ setFlag) |
        (setFlag === keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY
            ? keyring_trace_1.KeyringTraceFlag.DECRYPT_FLAGS
            : setFlag === keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY
                ? keyring_trace_1.KeyringTraceFlag.ENCRYPT_FLAGS
                : 0);
    let unencryptedDataKeyZeroed = false;
    let unencryptedDataKey;
    // This copy of the unencryptedDataKey is stored to insure that the
    // unencrypted data key is *never* modified.  Since the
    // unencryptedDataKey is returned by reference, any change
    // to it would be propagated to any cached versions.
    let udkForVerification;
    const setUnencryptedDataKey = (dataKey, trace) => {
        /* Avoid making unnecessary copies of the dataKey. */
        const tempUdk = dataKey instanceof Uint8Array ? dataKey : unwrapDataKey(dataKey);
        /* All security conditions are tested here and failures will throw. */
        verifyUnencryptedDataKeyForSet(tempUdk, trace);
        unencryptedDataKey = wrapWithKeyObjectIfSupported(dataKey);
        udkForVerification = new Uint8Array(tempUdk);
        material.keyringTrace.push(trace);
        return material;
    };
    const getUnencryptedDataKey = () => {
        /* Precondition: unencryptedDataKey must be set before we can return it. */
        (0, needs_1.needs)(unencryptedDataKey, 'unencryptedDataKey has not been set');
        /* Precondition: unencryptedDataKey must not be Zeroed out.
         * Returning a null key would be incredibly bad.
         */
        (0, needs_1.needs)(!unencryptedDataKeyZeroed, 'unencryptedDataKey has been zeroed.');
        /* Precondition: The unencryptedDataKey must not have been modified.
         * If the unencryptedDataKey is a KeyObject,
         * then the security around modification is handled in C.
         * Do not duplicate the secret just to check...
         */
        (0, needs_1.needs)(!(unencryptedDataKey instanceof Uint8Array) ||
            timingSafeEqual(udkForVerification, unwrapDataKey(unencryptedDataKey)), 'unencryptedDataKey has been corrupted.');
        return unencryptedDataKey;
    };
    Object.defineProperty(material, 'hasUnencryptedDataKey', {
        // Check that we have both not zeroed AND that we have not set
        get: () => !!unencryptedDataKey && !unencryptedDataKeyZeroed,
        enumerable: true,
    });
    const zeroUnencryptedDataKey = () => {
        /* These checks are separated on purpose.  It should be impossible to have only one unset.
         * *But* if it was the case, I *must* make sure I zero out the set one, and not leave it up to GC.
         * If I only checked on say unencryptedDataKey, and udkForVerification was somehow set,
         * doing the simplest thing would be to set both to new Uint8Array.
         * Leaving udkForVerification to be garbage collected.
         * This level of insanity is due to the fact that we are dealing with the unencrypted data key.
         */
        let unsetCount = 0;
        /* Precondition: If the unencryptedDataKey has not been set, it should not be settable later. */
        if (!unencryptedDataKey) {
            unencryptedDataKey = new Uint8Array();
            unsetCount += 1;
        }
        /* Precondition: If the udkForVerification has not been set, it should not be settable later. */
        if (!udkForVerification) {
            udkForVerification = new Uint8Array();
            unsetCount += 1;
        }
        /* The KeyObject manages its own ref counter.
         * Once there are no more users, it will clean the memory.
         */
        if (!(unencryptedDataKey instanceof Uint8Array)) {
            unencryptedDataKey = new Uint8Array();
        }
        unencryptedDataKey.fill(0);
        udkForVerification.fill(0);
        unencryptedDataKeyZeroed = true;
        /* Postcondition UNTESTED: Both unencryptedDataKey and udkForVerification must be either set or unset.
         * If it is ever the case that only one was unset, then something is wrong in a profound way.
         * It is not clear how this could ever happen, unless someone is manipulating the OS...
         */
        (0, needs_1.needs)(unsetCount === 0 || unsetCount === 2, 'Either unencryptedDataKey or udkForVerification was not set.');
        return material;
    };
    (0, immutable_class_1.readOnlyProperty)(material, 'setUnencryptedDataKey', setUnencryptedDataKey);
    (0, immutable_class_1.readOnlyProperty)(material, 'getUnencryptedDataKey', getUnencryptedDataKey);
    (0, immutable_class_1.readOnlyProperty)(material, 'zeroUnencryptedDataKey', zeroUnencryptedDataKey);
    return material;
    function verifyUnencryptedDataKeyForSet(dataKey, trace) {
        /* Precondition: unencryptedDataKey must not be set.  Modifying the unencryptedDataKey is denied */
        (0, needs_1.needs)(!unencryptedDataKey, 'unencryptedDataKey has already been set');
        /* Precondition: dataKey must be Binary Data */
        (0, needs_1.needs)(dataKey instanceof Uint8Array, 'dataKey must be a Uint8Array');
        /* Precondition: dataKey should have an ArrayBuffer that *only* stores the key.
         * This is a simple check to make sure that the key is not stored on
         * a large potentially shared ArrayBuffer.
         * If this was the case, it may be possible to find or manipulate.
         */
        (0, needs_1.needs)(dataKey.byteOffset === 0, 'Unencrypted Master Key must be an isolated buffer.');
        /* Precondition: The data key length must agree with algorithm specification.
         * If this is not the case, it either means ciphertext was tampered
         * with or the keyring implementation is not setting the length properly.
         */
        (0, needs_1.needs)(dataKey.byteLength === material.suite.keyLengthBytes, 'Key length does not agree with the algorithm specification.');
        /* Precondition: Trace must be set, and the flag must indicate that the data key was generated. */
        (0, needs_1.needs)(trace && trace.keyName && trace.keyNamespace, 'Malformed KeyringTrace');
        /* Precondition: On set the required KeyringTraceFlag must be set. */
        (0, needs_1.needs)(trace.flags & setFlag, 'Required KeyringTraceFlag not set');
        /* Precondition: Only valid flags are allowed.
         * An unencrypted data key can not be both generated and decrypted.
         */
        (0, needs_1.needs)(!(trace.flags & deniedSetFlags), 'Invalid KeyringTraceFlags set.');
    }
}
exports.decorateCryptographicMaterial = decorateCryptographicMaterial;
function decorateEncryptionMaterial(material) {
    const deniedEncryptFlags = keyring_trace_1.KeyringTraceFlag.SET_FLAGS | keyring_trace_1.KeyringTraceFlag.DECRYPT_FLAGS;
    const encryptedDataKeys = [];
    let signatureKey;
    const addEncryptedDataKey = (edk, flags) => {
        /* Precondition: If a data key has not already been generated, there must be no EDKs.
         * Pushing EDKs on the list before the data key has been generated may cause the list of
         * EDKs to be inconsistent. (i.e., they would decrypt to different data keys.)
         */
        (0, needs_1.needs)(material.hasUnencryptedDataKey, 'Unencrypted data key not set.');
        /* Precondition: Edk must be EncryptedDataKey
         * Putting things onto the list that are not EncryptedDataKey
         * may cause the list of EDKs to be inconsistent. (i.e. they may not serialize, or be mutable)
         */
        (0, needs_1.needs)(edk instanceof encrypted_data_key_1.EncryptedDataKey, 'Unsupported instance of encryptedDataKey');
        /* Precondition: flags must indicate that the key was encrypted. */
        (0, needs_1.needs)(flags & keyring_trace_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY, 'Encrypted data key flag must be set.');
        /* Precondition: flags must not include a setFlag or a decrypt flag.
         * The setFlag is reserved for setting the unencrypted data key
         * and must only occur once in the set of KeyringTrace flags.
         * The two setFlags in use are:
         * KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY
         * KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY
         *
         * KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX is reserved for the decrypt path
         */
        (0, needs_1.needs)(!(flags & deniedEncryptFlags), 'Invalid flag for EncryptedDataKey.');
        material.keyringTrace.push({
            keyName: edk.providerInfo,
            keyNamespace: edk.providerId,
            flags,
        });
        encryptedDataKeys.push(edk);
        return material;
    };
    (0, immutable_class_1.readOnlyProperty)(material, 'addEncryptedDataKey', addEncryptedDataKey);
    Object.defineProperty(material, 'encryptedDataKeys', {
        // I only want EDKs added through addEncryptedDataKey
        // so I return a new array
        get: () => [...encryptedDataKeys],
        enumerable: true,
    });
    const setSignatureKey = (key) => {
        /* Precondition: The SignatureKey stored must agree with the algorithm specification.
         * If this is not the case it means the MaterialManager or Keyring is not setting
         * the SignatureKey correctly
         */
        (0, needs_1.needs)(material.suite.signatureCurve, 'Algorithm specification does not support signatures.');
        /* Precondition: signatureKey must not be set.  Modifying the signatureKey is denied. */
        (0, needs_1.needs)(!signatureKey, 'Signature key has already been set.');
        /* Precondition: key must be a SignatureKey. */
        (0, needs_1.needs)(key instanceof signature_key_1.SignatureKey, 'Unsupported instance of key');
        signatureKey = key;
        return material;
    };
    (0, immutable_class_1.readOnlyProperty)(material, 'setSignatureKey', setSignatureKey);
    Object.defineProperty(material, 'signatureKey', {
        get: () => {
            /* Precondition: The SignatureKey requested must agree with the algorithm specification.
             * If this is not the case it means the MaterialManager or Keyring is not setting
             * the SignatureKey correctly
             */
            (0, needs_1.needs)(!!material.suite.signatureCurve === !!signatureKey, 'Algorithm specification not satisfied.');
            return signatureKey;
        },
        enumerable: true,
    });
    return material;
}
exports.decorateEncryptionMaterial = decorateEncryptionMaterial;
function decorateDecryptionMaterial(material) {
    // Verification Key
    let verificationKey;
    const setVerificationKey = (key) => {
        /* Precondition: The VerificationKey stored must agree with the algorithm specification.
         * If this is not the case it means the MaterialManager or Keyring is not setting
         * the VerificationKey correctly
         */
        (0, needs_1.needs)(material.suite.signatureCurve, 'Algorithm specification does not support signatures.');
        /* Precondition: verificationKey must not be set.  Modifying the verificationKey is denied. */
        (0, needs_1.needs)(!verificationKey, 'Verification key has already been set.');
        /* Precondition: key must be a VerificationKey. */
        (0, needs_1.needs)(key instanceof signature_key_1.VerificationKey, 'Unsupported instance of key');
        verificationKey = key;
        return material;
    };
    (0, immutable_class_1.readOnlyProperty)(material, 'setVerificationKey', setVerificationKey);
    Object.defineProperty(material, 'verificationKey', {
        get: () => {
            /* Precondition: The VerificationKey requested must agree with the algorithm specification.
             * If this is not the case it means the MaterialManager or Keyring is not setting
             * the VerificationKey correctly
             */
            (0, needs_1.needs)(!!material.suite.signatureCurve === !!verificationKey, 'Algorithm specification not satisfied.');
            return verificationKey;
        },
        enumerable: true,
    });
    return material;
}
exports.decorateDecryptionMaterial = decorateDecryptionMaterial;
function decorateWebCryptoMaterial(material, setFlags) {
    let cryptoKey;
    const setCryptoKey = (dataKey, trace) => {
        /* Precondition: cryptoKey must not be set.  Modifying the cryptoKey is denied */
        (0, needs_1.needs)(!cryptoKey, 'cryptoKey is already set.');
        /* Precondition: dataKey must be a supported type. */
        (0, needs_1.needs)(isCryptoKey(dataKey) || isMixedBackendCryptoKey(dataKey), 'Unsupported dataKey type.');
        /* Precondition: The CryptoKey must match the algorithm suite specification. */
        (0, needs_1.needs)(isValidCryptoKey(dataKey, material), 'CryptoKey settings not acceptable.');
        /* If the material does not have an unencrypted data key,
         * then we are setting the crypto key here and need a keyring trace .
         */
        if (!material.hasUnencryptedDataKey) {
            /* Precondition: If the CryptoKey is the only version, the trace information must be set here. */
            (0, needs_1.needs)(trace && trace.keyName && trace.keyNamespace, 'Malformed KeyringTrace');
            /* Precondition: On setting the CryptoKey the required KeyringTraceFlag must be set. */
            (0, needs_1.needs)(trace.flags & setFlags, 'Required KeyringTraceFlag not set');
            /* If I a setting a cryptoKey without an unencrypted data key,
             * an unencrypted data should never be set.
             * The expectation is if you are setting the cryptoKey *first* then
             * the unencrypted data key has already been "handled".
             * This ensures that a cryptoKey and an unencrypted data key always match.
             */
            material.zeroUnencryptedDataKey();
            material.keyringTrace.push(trace);
        }
        if (isCryptoKey(dataKey)) {
            cryptoKey = dataKey;
        }
        else {
            const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
            cryptoKey = Object.freeze({ zeroByteCryptoKey, nonZeroByteCryptoKey });
        }
        return material;
    };
    (0, immutable_class_1.readOnlyProperty)(material, 'setCryptoKey', setCryptoKey);
    const getCryptoKey = () => {
        /* Precondition: The cryptoKey must be set before we can return it. */
        (0, needs_1.needs)(cryptoKey, 'Crypto key is not set.');
        // In the case of MixedBackendCryptoKey the object
        // has already been frozen above so it is safe to return
        return cryptoKey;
    };
    (0, immutable_class_1.readOnlyProperty)(material, 'getCryptoKey', getCryptoKey);
    Object.defineProperty(material, 'hasCryptoKey', {
        get: () => !!cryptoKey,
        enumerable: true,
    });
    return material;
}
exports.decorateWebCryptoMaterial = decorateWebCryptoMaterial;
function isCryptoKey(dataKey) {
    return (dataKey &&
        'algorithm' in dataKey &&
        'type' in dataKey &&
        'usages' in dataKey &&
        'extractable' in dataKey);
}
exports.isCryptoKey = isCryptoKey;
function isValidCryptoKey(dataKey, material) {
    if (!isCryptoKey(dataKey)) {
        const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
        return (isValidCryptoKey(zeroByteCryptoKey, material) &&
            isValidCryptoKey(nonZeroByteCryptoKey, material));
    }
    const { suite, validUsages } = material;
    const { encryption, keyLength, kdf } = suite;
    /* See:
     * https://developer.mozilla.org/en-US/docs/Web/API/CryptoKey
     * https://developer.mozilla.org/en-US/docs/Web/API/AesKeyGenParams
     */
    const { type, algorithm, usages, extractable } = dataKey;
    // @ts-ignore length is an optional value...
    const { name, length } = algorithm;
    /* MSRCrypto, for legacy reasons,
     * normalizes the algorithm name
     * to lower case.
     * https://github.com/microsoft/MSR-JavaScript-Crypto/issues/1
     * For now, I'm going to upper case the name.
     */
    // Only symmetric algorithms
    return (type === 'secret' &&
        // Must match the suite
        ((kdf && name.toUpperCase() === kdf) ||
            (name.toUpperCase() === encryption && length === keyLength)) &&
        /* Only valid usage are: encrypt|decrypt|deriveKey
         * The complexity between deriveKey and suite.kdf should be handled in the Material class.
         */
        usages.some((u) => validUsages.includes(u)) &&
        // Since CryptoKey can not be zeroized, not extractable is the next best thing
        !extractable);
}
exports.isValidCryptoKey = isValidCryptoKey;
function isMixedBackendCryptoKey(dataKey) {
    const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
    return isCryptoKey(zeroByteCryptoKey) && isCryptoKey(nonZeroByteCryptoKey);
}
function keyUsageForMaterial(material) {
    const { suite } = material;
    if (suite.kdf)
        return 'deriveKey';
    return subtleFunctionForMaterial(material);
}
exports.keyUsageForMaterial = keyUsageForMaterial;
function subtleFunctionForMaterial(material) {
    if (material instanceof WebCryptoEncryptionMaterial)
        return 'encrypt';
    if (material instanceof WebCryptoDecryptionMaterial)
        return 'decrypt';
    throw new Error('Unsupported material');
}
exports.subtleFunctionForMaterial = subtleFunctionForMaterial;
function unwrapDataKey(dataKey) {
    if (dataKey instanceof Uint8Array)
        return dataKey;
    if (exports.supportsKeyObject && dataKey instanceof exports.supportsKeyObject.KeyObject)
        return dataKey.export();
    throw new Error('Unsupported dataKey type');
}
exports.unwrapDataKey = unwrapDataKey;
function wrapWithKeyObjectIfSupported(dataKey) {
    if (exports.supportsKeyObject) {
        if (dataKey instanceof Uint8Array) {
            const ko = exports.supportsKeyObject.createSecretKey(dataKey);
            /* Postcondition: Zero the secret.  It is now inside the KeyObject. */
            dataKey.fill(0);
            return ko;
        }
        if (dataKey instanceof exports.supportsKeyObject.KeyObject)
            return dataKey;
    }
    else if (dataKey instanceof Uint8Array) {
        return dataKey;
    }
    throw new Error('Unsupported dataKey type');
}
exports.wrapWithKeyObjectIfSupported = wrapWithKeyObjectIfSupported;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3J5cHRvZ3JhcGhpY19tYXRlcmlhbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9jcnlwdG9ncmFwaGljX21hdGVyaWFsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFXdEMsNkRBQXVEO0FBQ3ZELG1EQUErRDtBQUMvRCx1REFBaUU7QUFDakUsbURBQWdFO0FBQ2hFLHVEQUFzRDtBQUN0RCxtRUFBaUU7QUFDakUsbUNBQStCO0FBQy9CLCtCQUF3QztBQWUzQixRQUFBLGlCQUFpQixHQUFHLENBQUM7SUFDaEMsSUFBSTtRQUNGLE1BQU0sRUFBRSxTQUFTLEVBQUUsZUFBZSxFQUFFLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBa0IsQ0FBQSxDQUFDLHlEQUF5RDtRQUNuSSxJQUFJLENBQUMsU0FBUyxJQUFJLENBQUMsZUFBZTtZQUFFLE9BQU8sS0FBSyxDQUFBO1FBRWhELE9BQU8sRUFBRSxTQUFTLEVBQUUsZUFBZSxFQUFFLENBQUE7S0FDdEM7SUFBQyxPQUFPLEVBQUUsRUFBRTtRQUNYLE9BQU8sS0FBSyxDQUFBO0tBQ2I7QUFDSCxDQUFDLENBQUMsRUFBRSxDQUFBO0FBRUo7Ozs7Ozs7Ozs7Ozs7OztHQWVHO0FBRUgsTUFBTSxlQUFlLEdBQ25CLENBQUM7SUFDQyxJQUFJO1FBQ0Y7OztXQUdHO1FBQ0gsTUFBTSxFQUFFLGVBQWUsRUFBRSxtQkFBbUIsRUFBRSxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQSxDQUFDLHlEQUF5RDtRQUM1SCxPQUFPLG1CQUFtQixJQUFJLHVCQUF1QixDQUFBO0tBQ3REO0lBQUMsT0FBTyxDQUFDLEVBQUU7UUFDVixPQUFPLHVCQUF1QixDQUFBO0tBQy9CO0lBQ0Qsc0RBQXNEO0lBQ3RELFNBQVMsdUJBQXVCLENBQUMsQ0FBYSxFQUFFLENBQWE7UUFDM0Q7Ozs7Ozs7Ozs7V0FVRztRQUNILDBDQUEwQztRQUMxQzs7V0FFRztRQUNILElBQUksQ0FBQyxDQUFDLFVBQVUsS0FBSyxDQUFDLENBQUMsVUFBVTtZQUFFLE9BQU8sS0FBSyxDQUFBO1FBRS9DLElBQUksSUFBSSxHQUFHLENBQUMsQ0FBQTtRQUNaLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQ2pDLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO1NBQ3BCO1FBQ0QsT0FBTyxJQUFJLEtBQUssQ0FBQyxDQUFBO0lBQ25CLENBQUM7QUFDSCxDQUFDLENBQUMsRUFBRSxDQUFBO0FBa0NOLE1BQWEscUJBQXFCO0lBQ2hDLDJFQUEyRTtJQUMzRSwwQkFBMEI7SUFFMUIsOEVBQThFO0lBQzlFLGdCQUFnQjtJQUVoQiw4RUFBOEU7SUFDOUUsWUFBWTtJQUNKLFVBQVUsQ0FBUTtJQUsxQixZQUNFLFNBQWlCLEVBQ2pCLG1CQUEyQixFQUMzQixnQkFBd0IsRUFDeEIsaUJBQW9DO1FBRXBDLCtDQUErQztRQUMvQyxJQUFBLGFBQUssRUFBQyxTQUFTLFlBQVksTUFBTSxFQUFFLDZCQUE2QixDQUFDLENBQUE7UUFFakUsa0RBQWtEO1FBQ2xELElBQUEsYUFBSyxFQUNILE9BQU8sbUJBQW1CLEtBQUssUUFBUSxFQUN2QyxnQ0FBZ0MsQ0FDakMsQ0FBQTtRQUVELHVEQUF1RDtRQUN2RCxJQUFBLGFBQUssRUFDSCxPQUFPLGdCQUFnQixLQUFLLFFBQVEsRUFDcEMscUNBQXFDLENBQ3RDLENBQUE7UUFFRCw0RUFBNEU7UUFDNUUsSUFBQSxhQUFLLEVBQ0gsaUJBQWlCLElBQUksT0FBTyxpQkFBaUIsS0FBSyxRQUFRLEVBQzFELHNDQUFzQyxDQUN2QyxDQUFBO1FBRUQsMkRBQTJEO1FBQzNELElBQUEsYUFBSyxFQUFDLFNBQVMsQ0FBQyxNQUFNLEtBQUssRUFBRSxFQUFFLGtDQUFrQyxDQUFDLENBQUE7UUFFbEUsNkNBQTZDO1FBQzdDLElBQUEsYUFBSyxFQUFDLG1CQUFtQixFQUFFLHFCQUFxQixDQUFDLENBQUE7UUFFakQsbUVBQW1FO1FBQ25FLElBQUEsYUFBSyxFQUNILElBQUEsZUFBUSxFQUFDLGdCQUFnQixDQUFDLElBQUksSUFBQSxjQUFPLEVBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLEVBQzdELGlEQUFpRCxDQUNsRCxDQUFBO1FBRUQsNENBQTRDO1FBQzVDLElBQUksQ0FBQyxVQUFVLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQTtRQUV4QyxtREFBbUQ7UUFDbkQsSUFBSSxDQUFDLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxHQUFHLGlCQUFpQixFQUFFLENBQUMsQ0FBQTtRQUVoRSxJQUFJLENBQUMsbUJBQW1CLEdBQUcsbUJBQW1CLENBQUE7UUFFOUMsSUFBSSxDQUFDLGdCQUFnQixHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLENBQUE7UUFFOUQsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUscUJBQXFCLENBQUMsU0FBUyxDQUFDLENBQUE7UUFDNUQsdUNBQXVDO1FBQ3ZDLDZEQUE2RDtRQUM3RCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFBO0lBQ3JCLENBQUM7SUFFRCxtRUFBbUU7SUFDbkUsU0FBUztRQUNQLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQTtJQUN4QixDQUFDO0lBRUQsMkVBQTJFO0lBQzNFLHlFQUF5RTtJQUN6RSxvREFBb0Q7SUFDcEQsc0JBQXNCO1FBQ3BCLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFBO1FBQ3ZCLE9BQU8sSUFBSSxDQUFBO0lBQ2IsQ0FBQztDQUNGO0FBakZELHNEQWlGQztBQUNELDJCQUEyQjtBQUMzQixJQUFBLDZCQUFXLEVBQUMscUJBQXFCLENBQUMsQ0FBQTtBQTJCbEMsTUFBYSxzQkFBc0I7SUFLakMsS0FBSyxDQUFvQjtJQUN6QixxQkFBcUIsQ0FHTTtJQUMzQixxQkFBcUIsQ0FBc0M7SUFDM0Qsc0JBQXNCLENBQStCO0lBQ3JELHFCQUFxQixDQUFVO0lBQy9CLFlBQVksR0FBbUIsRUFBRSxDQUFBO0lBQ2pDLGlCQUFpQixDQUFxQjtJQUN0QyxtQkFBbUIsQ0FHUTtJQUMzQixlQUFlLENBQWdEO0lBQy9ELFlBQVksQ0FBZTtJQUMzQixpQkFBaUIsQ0FBNkI7SUFDOUMsWUFBWSxLQUF5QixFQUFFLGlCQUFvQztRQUN6RSw0RUFBNEU7UUFDNUUsSUFBQSxhQUFLLEVBQ0gsS0FBSyxZQUFZLG9DQUFrQixFQUNuQyxvQ0FBb0MsQ0FDckMsQ0FBQTtRQUNELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFBO1FBQ2xCLG9HQUFvRztRQUNwRyxJQUFBLGFBQUssRUFDSCxpQkFBaUIsSUFBSSxPQUFPLGlCQUFpQixLQUFLLFFBQVEsRUFDMUQsZ0NBQWdDLENBQ2pDLENBQUE7UUFDRCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEdBQUcsaUJBQWlCLEVBQUUsQ0FBQyxDQUFBO1FBQ2hFLHdFQUF3RTtRQUN4RSxNQUFNLFFBQVEsR0FBRyxnQ0FBZ0IsQ0FBQywrQkFBK0IsQ0FBQTtRQUNqRSw2QkFBNkIsQ0FBeUIsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFBO1FBQ3JFLDBCQUEwQixDQUF5QixJQUFJLENBQUMsQ0FBQTtRQUN4RCxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQyxTQUFTLENBQUMsQ0FBQTtRQUM3RCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFBO0lBQ3JCLENBQUM7SUFDRCxXQUFXO1FBQ1QsT0FBTyxJQUFJLENBQUMscUJBQXFCLENBQUE7SUFDbkMsQ0FBQztDQUNGO0FBN0NELHdEQTZDQztBQUNELElBQUEsNkJBQVcsRUFBQyxzQkFBc0IsQ0FBQyxDQUFBO0FBRW5DLE1BQWEsc0JBQXNCO0lBS2pDLEtBQUssQ0FBb0I7SUFDekIscUJBQXFCLENBR007SUFDM0IscUJBQXFCLENBQXNDO0lBQzNELHNCQUFzQixDQUErQjtJQUNyRCxxQkFBcUIsQ0FBVTtJQUMvQixZQUFZLEdBQW1CLEVBQUUsQ0FBQTtJQUNqQyxrQkFBa0IsQ0FBbUQ7SUFDckUsZUFBZSxDQUFrQjtJQUNqQyxpQkFBaUIsQ0FBNkI7SUFDOUMsWUFBWSxLQUF5QixFQUFFLGlCQUFvQztRQUN6RSw0RUFBNEU7UUFDNUUsSUFBQSxhQUFLLEVBQ0gsS0FBSyxZQUFZLG9DQUFrQixFQUNuQyxvQ0FBb0MsQ0FDckMsQ0FBQTtRQUNELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFBO1FBQ2xCLG9HQUFvRztRQUNwRyxJQUFBLGFBQUssRUFDSCxpQkFBaUIsSUFBSSxPQUFPLGlCQUFpQixLQUFLLFFBQVEsRUFDMUQsZ0NBQWdDLENBQ2pDLENBQUE7UUFDRCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEdBQUcsaUJBQWlCLEVBQUUsQ0FBQyxDQUFBO1FBQ2hFLHdFQUF3RTtRQUN4RSxNQUFNLFFBQVEsR0FBRyxnQ0FBZ0IsQ0FBQywrQkFBK0IsQ0FBQTtRQUNqRSw2QkFBNkIsQ0FBeUIsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFBO1FBQ3JFLDBCQUEwQixDQUF5QixJQUFJLENBQUMsQ0FBQTtRQUN4RCxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQyxTQUFTLENBQUMsQ0FBQTtRQUM3RCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFBO0lBQ3JCLENBQUM7SUFDRCxXQUFXO1FBQ1QsT0FBTyxJQUFJLENBQUMscUJBQXFCLENBQUE7SUFDbkMsQ0FBQztDQUNGO0FBeENELHdEQXdDQztBQUNELElBQUEsNkJBQVcsRUFBQyxzQkFBc0IsQ0FBQyxDQUFBO0FBRW5DLE1BQWEsMkJBQTJCO0lBTXRDLEtBQUssQ0FBeUI7SUFDOUIscUJBQXFCLENBR1c7SUFDaEMscUJBQXFCLENBQXNDO0lBQzNELHNCQUFzQixDQUFvQztJQUMxRCxxQkFBcUIsQ0FBVTtJQUMvQixZQUFZLEdBQW1CLEVBQUUsQ0FBQTtJQUNqQyxpQkFBaUIsQ0FBcUI7SUFDdEMsbUJBQW1CLENBR2E7SUFDaEMsZUFBZSxDQUFxRDtJQUNwRSxZQUFZLENBQWU7SUFDM0IsWUFBWSxDQUdvQjtJQUNoQyxZQUFZLENBQW1EO0lBQy9ELFlBQVksQ0FBVTtJQUN0QixXQUFXLENBQWtDO0lBQzdDLGlCQUFpQixDQUE2QjtJQUM5QyxZQUNFLEtBQThCLEVBQzlCLGlCQUFvQztRQUVwQyxzRkFBc0Y7UUFDdEYsSUFBQSxhQUFLLEVBQ0gsS0FBSyxZQUFZLCtDQUF1QixFQUN4Qyx5Q0FBeUMsQ0FDMUMsQ0FBQTtRQUNELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFBO1FBQ2xCLElBQUksQ0FBQyxXQUFXLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQztZQUMvQixXQUFXO1lBQ1gsU0FBUztTQUNhLENBQUMsQ0FBQTtRQUN6Qix5R0FBeUc7UUFDekcsSUFBQSxhQUFLLEVBQ0gsaUJBQWlCLElBQUksT0FBTyxpQkFBaUIsS0FBSyxRQUFRLEVBQzFELGdDQUFnQyxDQUNqQyxDQUFBO1FBQ0QsSUFBSSxDQUFDLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxHQUFHLGlCQUFpQixFQUFFLENBQUMsQ0FBQTtRQUNoRSx3RUFBd0U7UUFDeEUsTUFBTSxPQUFPLEdBQUcsZ0NBQWdCLENBQUMsK0JBQStCLENBQUE7UUFDaEUsNkJBQTZCLENBQThCLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQTtRQUN6RSwwQkFBMEIsQ0FBOEIsSUFBSSxDQUFDLENBQUE7UUFDN0QseUJBQXlCLENBQThCLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQTtRQUNyRSxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSwyQkFBMkIsQ0FBQyxTQUFTLENBQUMsQ0FBQTtRQUNsRSxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFBO0lBQ3JCLENBQUM7SUFDRCxXQUFXO1FBQ1QsT0FBTyxJQUFJLENBQUMscUJBQXFCLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQTtJQUN4RCxDQUFDO0NBQ0Y7QUE3REQsa0VBNkRDO0FBQ0QsSUFBQSw2QkFBVyxFQUFDLDJCQUEyQixDQUFDLENBQUE7QUFFeEMsTUFBYSwyQkFBMkI7SUFNdEMsS0FBSyxDQUF5QjtJQUM5QixxQkFBcUIsQ0FHVztJQUNoQyxxQkFBcUIsQ0FBc0M7SUFDM0Qsc0JBQXNCLENBQW9DO0lBQzFELHFCQUFxQixDQUFVO0lBQy9CLFlBQVksR0FBbUIsRUFBRSxDQUFBO0lBQ2pDLGtCQUFrQixDQUF3RDtJQUMxRSxlQUFlLENBQWtCO0lBQ2pDLFlBQVksQ0FHb0I7SUFDaEMsWUFBWSxDQUFtRDtJQUMvRCxZQUFZLENBQVU7SUFDdEIsV0FBVyxDQUFrQztJQUM3QyxpQkFBaUIsQ0FBNkI7SUFDOUMsWUFDRSxLQUE4QixFQUM5QixpQkFBb0M7UUFFcEMsc0ZBQXNGO1FBQ3RGLElBQUEsYUFBSyxFQUNILEtBQUssWUFBWSwrQ0FBdUIsRUFDeEMseUNBQXlDLENBQzFDLENBQUE7UUFDRCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQTtRQUNsQixJQUFJLENBQUMsV0FBVyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUM7WUFDL0IsV0FBVztZQUNYLFNBQVM7U0FDYSxDQUFDLENBQUE7UUFDekIseUdBQXlHO1FBQ3pHLElBQUEsYUFBSyxFQUNILGlCQUFpQixJQUFJLE9BQU8saUJBQWlCLEtBQUssUUFBUSxFQUMxRCxnQ0FBZ0MsQ0FDakMsQ0FBQTtRQUNELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsR0FBRyxpQkFBaUIsRUFBRSxDQUFDLENBQUE7UUFDaEUsd0VBQXdFO1FBQ3hFLE1BQU0sT0FBTyxHQUFHLGdDQUFnQixDQUFDLCtCQUErQixDQUFBO1FBQ2hFLDZCQUE2QixDQUE4QixJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUE7UUFDekUsMEJBQTBCLENBQThCLElBQUksQ0FBQyxDQUFBO1FBQzdELHlCQUF5QixDQUE4QixJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUE7UUFDckUsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLENBQUMsU0FBUyxDQUFDLENBQUE7UUFDbEUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQTtJQUNyQixDQUFDO0lBQ0QsV0FBVztRQUNULE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQTtJQUMxQixDQUFDO0NBQ0Y7QUF4REQsa0VBd0RDO0FBQ0QsSUFBQSw2QkFBVyxFQUFDLDJCQUEyQixDQUFDLENBQUE7QUFFeEMsU0FBZ0Isb0JBQW9CLENBQ2xDLEdBQVE7SUFFUixPQUFPLENBQ0wsR0FBRyxZQUFZLDJCQUEyQjtRQUMxQyxHQUFHLFlBQVksc0JBQXNCLENBQ3RDLENBQUE7QUFDSCxDQUFDO0FBUEQsb0RBT0M7QUFFRCxTQUFnQixvQkFBb0IsQ0FDbEMsR0FBUTtJQUVSLE9BQU8sQ0FDTCxHQUFHLFlBQVksMkJBQTJCO1FBQzFDLEdBQUcsWUFBWSxzQkFBc0IsQ0FDdEMsQ0FBQTtBQUNILENBQUM7QUFQRCxvREFPQztBQUVELFNBQWdCLG1CQUFtQixDQUFDLEdBQVE7SUFDMUMsT0FBTyxHQUFHLFlBQVkscUJBQXFCLENBQUE7QUFDN0MsQ0FBQztBQUZELGtEQUVDO0FBRUQsU0FBZ0IsNkJBQTZCLENBRTNDLFFBQVcsRUFBRSxPQUF5QjtJQUN0Qyw2RUFBNkU7SUFDN0UsSUFBQSxhQUFLLEVBQUMsT0FBTyxHQUFHLGdDQUFnQixDQUFDLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxDQUFBO0lBQzlEOzs7T0FHRztJQUNILE1BQU0sY0FBYyxHQUNsQixDQUFDLGdDQUFnQixDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUM7UUFDdEMsQ0FBQyxPQUFPLEtBQUssZ0NBQWdCLENBQUMsK0JBQStCO1lBQzNELENBQUMsQ0FBQyxnQ0FBZ0IsQ0FBQyxhQUFhO1lBQ2hDLENBQUMsQ0FBQyxPQUFPLEtBQUssZ0NBQWdCLENBQUMsK0JBQStCO2dCQUM5RCxDQUFDLENBQUMsZ0NBQWdCLENBQUMsYUFBYTtnQkFDaEMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO0lBRVIsSUFBSSx3QkFBd0IsR0FBRyxLQUFLLENBQUE7SUFDcEMsSUFBSSxrQkFBaUQsQ0FBQTtJQUNyRCxtRUFBbUU7SUFDbkUsdURBQXVEO0lBQ3ZELDBEQUEwRDtJQUMxRCxvREFBb0Q7SUFDcEQsSUFBSSxrQkFBOEIsQ0FBQTtJQUVsQyxNQUFNLHFCQUFxQixHQUFHLENBQzVCLE9BQXNDLEVBQ3RDLEtBQW1CLEVBQ25CLEVBQUU7UUFDRixxREFBcUQ7UUFDckQsTUFBTSxPQUFPLEdBQ1gsT0FBTyxZQUFZLFVBQVUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUE7UUFDbEUsc0VBQXNFO1FBQ3RFLDhCQUE4QixDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQTtRQUM5QyxrQkFBa0IsR0FBRyw0QkFBNEIsQ0FBQyxPQUFPLENBQUMsQ0FBQTtRQUMxRCxrQkFBa0IsR0FBRyxJQUFJLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQTtRQUM1QyxRQUFRLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQTtRQUVqQyxPQUFPLFFBQVEsQ0FBQTtJQUNqQixDQUFDLENBQUE7SUFDRCxNQUFNLHFCQUFxQixHQUFHLEdBQWtDLEVBQUU7UUFDaEUsMkVBQTJFO1FBQzNFLElBQUEsYUFBSyxFQUFDLGtCQUFrQixFQUFFLHFDQUFxQyxDQUFDLENBQUE7UUFDaEU7O1dBRUc7UUFDSCxJQUFBLGFBQUssRUFBQyxDQUFDLHdCQUF3QixFQUFFLHFDQUFxQyxDQUFDLENBQUE7UUFDdkU7Ozs7V0FJRztRQUNILElBQUEsYUFBSyxFQUNILENBQUMsQ0FBQyxrQkFBa0IsWUFBWSxVQUFVLENBQUM7WUFDekMsZUFBZSxDQUFDLGtCQUFrQixFQUFFLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLEVBQ3hFLHdDQUF3QyxDQUN6QyxDQUFBO1FBQ0QsT0FBTyxrQkFBa0IsQ0FBQTtJQUMzQixDQUFDLENBQUE7SUFDRCxNQUFNLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSx1QkFBdUIsRUFBRTtRQUN2RCw4REFBOEQ7UUFDOUQsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsSUFBSSxDQUFDLHdCQUF3QjtRQUM1RCxVQUFVLEVBQUUsSUFBSTtLQUNqQixDQUFDLENBQUE7SUFDRixNQUFNLHNCQUFzQixHQUFHLEdBQUcsRUFBRTtRQUNsQzs7Ozs7O1dBTUc7UUFDSCxJQUFJLFVBQVUsR0FBRyxDQUFDLENBQUE7UUFDbEIsZ0dBQWdHO1FBQ2hHLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUN2QixrQkFBa0IsR0FBRyxJQUFJLFVBQVUsRUFBRSxDQUFBO1lBQ3JDLFVBQVUsSUFBSSxDQUFDLENBQUE7U0FDaEI7UUFDRCxnR0FBZ0c7UUFDaEcsSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQ3ZCLGtCQUFrQixHQUFHLElBQUksVUFBVSxFQUFFLENBQUE7WUFDckMsVUFBVSxJQUFJLENBQUMsQ0FBQTtTQUNoQjtRQUNEOztXQUVHO1FBQ0gsSUFBSSxDQUFDLENBQUMsa0JBQWtCLFlBQVksVUFBVSxDQUFDLEVBQUU7WUFDL0Msa0JBQWtCLEdBQUcsSUFBSSxVQUFVLEVBQUUsQ0FBQTtTQUN0QztRQUNELGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQTtRQUMxQixrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUE7UUFDMUIsd0JBQXdCLEdBQUcsSUFBSSxDQUFBO1FBRS9COzs7V0FHRztRQUNILElBQUEsYUFBSyxFQUNILFVBQVUsS0FBSyxDQUFDLElBQUksVUFBVSxLQUFLLENBQUMsRUFDcEMsOERBQThELENBQy9ELENBQUE7UUFDRCxPQUFPLFFBQVEsQ0FBQTtJQUNqQixDQUFDLENBQUE7SUFFRCxJQUFBLGtDQUFnQixFQUFDLFFBQVEsRUFBRSx1QkFBdUIsRUFBRSxxQkFBcUIsQ0FBQyxDQUFBO0lBQzFFLElBQUEsa0NBQWdCLEVBQUMsUUFBUSxFQUFFLHVCQUF1QixFQUFFLHFCQUFxQixDQUFDLENBQUE7SUFDMUUsSUFBQSxrQ0FBZ0IsRUFBQyxRQUFRLEVBQUUsd0JBQXdCLEVBQUUsc0JBQXNCLENBQUMsQ0FBQTtJQUU1RSxPQUFPLFFBQVEsQ0FBQTtJQUVmLFNBQVMsOEJBQThCLENBQ3JDLE9BQW1CLEVBQ25CLEtBQW1CO1FBRW5CLG1HQUFtRztRQUNuRyxJQUFBLGFBQUssRUFBQyxDQUFDLGtCQUFrQixFQUFFLHlDQUF5QyxDQUFDLENBQUE7UUFDckUsK0NBQStDO1FBQy9DLElBQUEsYUFBSyxFQUFDLE9BQU8sWUFBWSxVQUFVLEVBQUUsOEJBQThCLENBQUMsQ0FBQTtRQUNwRTs7OztXQUlHO1FBQ0gsSUFBQSxhQUFLLEVBQ0gsT0FBTyxDQUFDLFVBQVUsS0FBSyxDQUFDLEVBQ3hCLG9EQUFvRCxDQUNyRCxDQUFBO1FBQ0Q7OztXQUdHO1FBQ0gsSUFBQSxhQUFLLEVBQ0gsT0FBTyxDQUFDLFVBQVUsS0FBSyxRQUFRLENBQUMsS0FBSyxDQUFDLGNBQWMsRUFDcEQsNkRBQTZELENBQzlELENBQUE7UUFFRCxrR0FBa0c7UUFDbEcsSUFBQSxhQUFLLEVBQ0gsS0FBSyxJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLFlBQVksRUFDNUMsd0JBQXdCLENBQ3pCLENBQUE7UUFDRCxxRUFBcUU7UUFDckUsSUFBQSxhQUFLLEVBQUMsS0FBSyxDQUFDLEtBQUssR0FBRyxPQUFPLEVBQUUsbUNBQW1DLENBQUMsQ0FBQTtRQUNqRTs7V0FFRztRQUNILElBQUEsYUFBSyxFQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsS0FBSyxHQUFHLGNBQWMsQ0FBQyxFQUFFLGdDQUFnQyxDQUFDLENBQUE7SUFDMUUsQ0FBQztBQUNILENBQUM7QUFwSkQsc0VBb0pDO0FBRUQsU0FBZ0IsMEJBQTBCLENBQ3hDLFFBQVc7SUFFWCxNQUFNLGtCQUFrQixHQUN0QixnQ0FBZ0IsQ0FBQyxTQUFTLEdBQUcsZ0NBQWdCLENBQUMsYUFBYSxDQUFBO0lBQzdELE1BQU0saUJBQWlCLEdBQXVCLEVBQUUsQ0FBQTtJQUNoRCxJQUFJLFlBQWdELENBQUE7SUFFcEQsTUFBTSxtQkFBbUIsR0FBRyxDQUMxQixHQUFxQixFQUNyQixLQUF1QixFQUN2QixFQUFFO1FBQ0Y7OztXQUdHO1FBQ0gsSUFBQSxhQUFLLEVBQUMsUUFBUSxDQUFDLHFCQUFxQixFQUFFLCtCQUErQixDQUFDLENBQUE7UUFDdEU7OztXQUdHO1FBQ0gsSUFBQSxhQUFLLEVBQ0gsR0FBRyxZQUFZLHFDQUFnQixFQUMvQiwwQ0FBMEMsQ0FDM0MsQ0FBQTtRQUVELG1FQUFtRTtRQUNuRSxJQUFBLGFBQUssRUFDSCxLQUFLLEdBQUcsZ0NBQWdCLENBQUMsK0JBQStCLEVBQ3hELHNDQUFzQyxDQUN2QyxDQUFBO1FBRUQ7Ozs7Ozs7O1dBUUc7UUFDSCxJQUFBLGFBQUssRUFBQyxDQUFDLENBQUMsS0FBSyxHQUFHLGtCQUFrQixDQUFDLEVBQUUsb0NBQW9DLENBQUMsQ0FBQTtRQUMxRSxRQUFRLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQztZQUN6QixPQUFPLEVBQUUsR0FBRyxDQUFDLFlBQVk7WUFDekIsWUFBWSxFQUFFLEdBQUcsQ0FBQyxVQUFVO1lBQzVCLEtBQUs7U0FDTixDQUFDLENBQUE7UUFFRixpQkFBaUIsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUE7UUFDM0IsT0FBTyxRQUFRLENBQUE7SUFDakIsQ0FBQyxDQUFBO0lBRUQsSUFBQSxrQ0FBZ0IsRUFBQyxRQUFRLEVBQUUscUJBQXFCLEVBQUUsbUJBQW1CLENBQUMsQ0FBQTtJQUN0RSxNQUFNLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxtQkFBbUIsRUFBRTtRQUNuRCxxREFBcUQ7UUFDckQsMEJBQTBCO1FBQzFCLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEdBQUcsaUJBQWlCLENBQUM7UUFDakMsVUFBVSxFQUFFLElBQUk7S0FDakIsQ0FBQyxDQUFBO0lBQ0YsTUFBTSxlQUFlLEdBQUcsQ0FBQyxHQUFpQixFQUFFLEVBQUU7UUFDNUM7OztXQUdHO1FBQ0gsSUFBQSxhQUFLLEVBQ0gsUUFBUSxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQzdCLHNEQUFzRCxDQUN2RCxDQUFBO1FBQ0Qsd0ZBQXdGO1FBQ3hGLElBQUEsYUFBSyxFQUFDLENBQUMsWUFBWSxFQUFFLHFDQUFxQyxDQUFDLENBQUE7UUFDM0QsK0NBQStDO1FBQy9DLElBQUEsYUFBSyxFQUFDLEdBQUcsWUFBWSw0QkFBWSxFQUFFLDZCQUE2QixDQUFDLENBQUE7UUFDakUsWUFBWSxHQUFHLEdBQUcsQ0FBQTtRQUNsQixPQUFPLFFBQVEsQ0FBQTtJQUNqQixDQUFDLENBQUE7SUFDRCxJQUFBLGtDQUFnQixFQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRSxlQUFlLENBQUMsQ0FBQTtJQUM5RCxNQUFNLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxjQUFjLEVBQUU7UUFDOUMsR0FBRyxFQUFFLEdBQUcsRUFBRTtZQUNSOzs7ZUFHRztZQUNILElBQUEsYUFBSyxFQUNILENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGNBQWMsS0FBSyxDQUFDLENBQUMsWUFBWSxFQUNsRCx3Q0FBd0MsQ0FDekMsQ0FBQTtZQUNELE9BQU8sWUFBWSxDQUFBO1FBQ3JCLENBQUM7UUFDRCxVQUFVLEVBQUUsSUFBSTtLQUNqQixDQUFDLENBQUE7SUFFRixPQUFPLFFBQVEsQ0FBQTtBQUNqQixDQUFDO0FBNUZELGdFQTRGQztBQUVELFNBQWdCLDBCQUEwQixDQUN4QyxRQUFXO0lBRVgsbUJBQW1CO0lBQ25CLElBQUksZUFBc0QsQ0FBQTtJQUMxRCxNQUFNLGtCQUFrQixHQUFHLENBQUMsR0FBb0IsRUFBRSxFQUFFO1FBQ2xEOzs7V0FHRztRQUNILElBQUEsYUFBSyxFQUNILFFBQVEsQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUM3QixzREFBc0QsQ0FDdkQsQ0FBQTtRQUNELDhGQUE4RjtRQUM5RixJQUFBLGFBQUssRUFBQyxDQUFDLGVBQWUsRUFBRSx3Q0FBd0MsQ0FBQyxDQUFBO1FBQ2pFLGtEQUFrRDtRQUNsRCxJQUFBLGFBQUssRUFBQyxHQUFHLFlBQVksK0JBQWUsRUFBRSw2QkFBNkIsQ0FBQyxDQUFBO1FBQ3BFLGVBQWUsR0FBRyxHQUFHLENBQUE7UUFDckIsT0FBTyxRQUFRLENBQUE7SUFDakIsQ0FBQyxDQUFBO0lBQ0QsSUFBQSxrQ0FBZ0IsRUFBQyxRQUFRLEVBQUUsb0JBQW9CLEVBQUUsa0JBQWtCLENBQUMsQ0FBQTtJQUNwRSxNQUFNLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRTtRQUNqRCxHQUFHLEVBQUUsR0FBRyxFQUFFO1lBQ1I7OztlQUdHO1lBQ0gsSUFBQSxhQUFLLEVBQ0gsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsY0FBYyxLQUFLLENBQUMsQ0FBQyxlQUFlLEVBQ3JELHdDQUF3QyxDQUN6QyxDQUFBO1lBQ0QsT0FBTyxlQUFlLENBQUE7UUFDeEIsQ0FBQztRQUNELFVBQVUsRUFBRSxJQUFJO0tBQ2pCLENBQUMsQ0FBQTtJQUVGLE9BQU8sUUFBUSxDQUFBO0FBQ2pCLENBQUM7QUF0Q0QsZ0VBc0NDO0FBRUQsU0FBZ0IseUJBQXlCLENBQ3ZDLFFBQVcsRUFDWCxRQUEwQjtJQUUxQixJQUFJLFNBRVMsQ0FBQTtJQUViLE1BQU0sWUFBWSxHQUFHLENBQ25CLE9BQW1ELEVBQ25ELEtBQW1CLEVBQ25CLEVBQUU7UUFDRixpRkFBaUY7UUFDakYsSUFBQSxhQUFLLEVBQUMsQ0FBQyxTQUFTLEVBQUUsMkJBQTJCLENBQUMsQ0FBQTtRQUM5QyxxREFBcUQ7UUFDckQsSUFBQSxhQUFLLEVBQ0gsV0FBVyxDQUFDLE9BQU8sQ0FBQyxJQUFJLHVCQUF1QixDQUFDLE9BQU8sQ0FBQyxFQUN4RCwyQkFBMkIsQ0FDNUIsQ0FBQTtRQUNELCtFQUErRTtRQUMvRSxJQUFBLGFBQUssRUFDSCxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLEVBQ25DLG9DQUFvQyxDQUNyQyxDQUFBO1FBRUQ7O1dBRUc7UUFDSCxJQUFJLENBQUMsUUFBUSxDQUFDLHFCQUFxQixFQUFFO1lBQ25DLGlHQUFpRztZQUNqRyxJQUFBLGFBQUssRUFDSCxLQUFLLElBQUksS0FBSyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUMsWUFBWSxFQUM1Qyx3QkFBd0IsQ0FDekIsQ0FBQTtZQUNELHVGQUF1RjtZQUN2RixJQUFBLGFBQUssRUFBQyxLQUFLLENBQUMsS0FBSyxHQUFHLFFBQVEsRUFBRSxtQ0FBbUMsQ0FBQyxDQUFBO1lBQ2xFOzs7OztlQUtHO1lBQ0gsUUFBUSxDQUFDLHNCQUFzQixFQUFFLENBQUE7WUFDakMsUUFBUSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUE7U0FDbEM7UUFFRCxJQUFJLFdBQVcsQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUN4QixTQUFTLEdBQUcsT0FBTyxDQUFBO1NBQ3BCO2FBQU07WUFDTCxNQUFNLEVBQUUsaUJBQWlCLEVBQUUsb0JBQW9CLEVBQUUsR0FBRyxPQUFPLENBQUE7WUFDM0QsU0FBUyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxvQkFBb0IsRUFBRSxDQUFDLENBQUE7U0FDdkU7UUFFRCxPQUFPLFFBQVEsQ0FBQTtJQUNqQixDQUFDLENBQUE7SUFFRCxJQUFBLGtDQUFnQixFQUFDLFFBQVEsRUFBRSxjQUFjLEVBQUUsWUFBWSxDQUFDLENBQUE7SUFDeEQsTUFBTSxZQUFZLEdBQUcsR0FBRyxFQUFFO1FBQ3hCLHNFQUFzRTtRQUN0RSxJQUFBLGFBQUssRUFBQyxTQUFTLEVBQUUsd0JBQXdCLENBQUMsQ0FBQTtRQUMxQyxrREFBa0Q7UUFDbEQsd0RBQXdEO1FBQ3hELE9BQU8sU0FBaUUsQ0FBQTtJQUMxRSxDQUFDLENBQUE7SUFDRCxJQUFBLGtDQUFnQixFQUFDLFFBQVEsRUFBRSxjQUFjLEVBQUUsWUFBWSxDQUFDLENBQUE7SUFFeEQsTUFBTSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUUsY0FBYyxFQUFFO1FBQzlDLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztRQUN0QixVQUFVLEVBQUUsSUFBSTtLQUNqQixDQUFDLENBQUE7SUFFRixPQUFPLFFBQVEsQ0FBQTtBQUNqQixDQUFDO0FBeEVELDhEQXdFQztBQUVELFNBQWdCLFdBQVcsQ0FBQyxPQUFZO0lBQ3RDLE9BQU8sQ0FDTCxPQUFPO1FBQ1AsV0FBVyxJQUFJLE9BQU87UUFDdEIsTUFBTSxJQUFJLE9BQU87UUFDakIsUUFBUSxJQUFJLE9BQU87UUFDbkIsYUFBYSxJQUFJLE9BQU8sQ0FDekIsQ0FBQTtBQUNILENBQUM7QUFSRCxrQ0FRQztBQUVELFNBQWdCLGdCQUFnQixDQUM5QixPQUFtRCxFQUNuRCxRQUFXO0lBRVgsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsRUFBRTtRQUN6QixNQUFNLEVBQUUsaUJBQWlCLEVBQUUsb0JBQW9CLEVBQUUsR0FBRyxPQUFPLENBQUE7UUFDM0QsT0FBTyxDQUNMLGdCQUFnQixDQUFDLGlCQUFpQixFQUFFLFFBQVEsQ0FBQztZQUM3QyxnQkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRSxRQUFRLENBQUMsQ0FDakQsQ0FBQTtLQUNGO0lBRUQsTUFBTSxFQUFFLEtBQUssRUFBRSxXQUFXLEVBQUUsR0FBRyxRQUFRLENBQUE7SUFDdkMsTUFBTSxFQUFFLFVBQVUsRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLEdBQUcsS0FBSyxDQUFBO0lBRTVDOzs7T0FHRztJQUVILE1BQU0sRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsR0FBRyxPQUFPLENBQUE7SUFDeEQsNENBQTRDO0lBQzVDLE1BQU0sRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLEdBQUcsU0FBUyxDQUFBO0lBRWxDOzs7OztPQUtHO0lBRUgsNEJBQTRCO0lBQzVCLE9BQU8sQ0FDTCxJQUFJLEtBQUssUUFBUTtRQUNqQix1QkFBdUI7UUFDdkIsQ0FBQyxDQUFDLEdBQUcsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssR0FBRyxDQUFDO1lBQ2xDLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLLFVBQVUsSUFBSSxNQUFNLEtBQUssU0FBUyxDQUFDLENBQUM7UUFDOUQ7O1dBRUc7UUFDSCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzNDLDhFQUE4RTtRQUM5RSxDQUFDLFdBQVcsQ0FDYixDQUFBO0FBQ0gsQ0FBQztBQTVDRCw0Q0E0Q0M7QUFFRCxTQUFTLHVCQUF1QixDQUM5QixPQUFZO0lBRVosTUFBTSxFQUFFLGlCQUFpQixFQUFFLG9CQUFvQixFQUFFLEdBQUcsT0FBTyxDQUFBO0lBQzNELE9BQU8sV0FBVyxDQUFDLGlCQUFpQixDQUFDLElBQUksV0FBVyxDQUFDLG9CQUFvQixDQUFDLENBQUE7QUFDNUUsQ0FBQztBQUVELFNBQWdCLG1CQUFtQixDQUNqQyxRQUFXO0lBRVgsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLFFBQVEsQ0FBQTtJQUMxQixJQUFJLEtBQUssQ0FBQyxHQUFHO1FBQUUsT0FBTyxXQUFXLENBQUE7SUFDakMsT0FBTyx5QkFBeUIsQ0FBQyxRQUFRLENBQUMsQ0FBQTtBQUM1QyxDQUFDO0FBTkQsa0RBTUM7QUFFRCxTQUFnQix5QkFBeUIsQ0FDdkMsUUFBVztJQUVYLElBQUksUUFBUSxZQUFZLDJCQUEyQjtRQUFFLE9BQU8sU0FBUyxDQUFBO0lBQ3JFLElBQUksUUFBUSxZQUFZLDJCQUEyQjtRQUFFLE9BQU8sU0FBUyxDQUFBO0lBRXJFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsQ0FBQTtBQUN6QyxDQUFDO0FBUEQsOERBT0M7QUFFRCxTQUFnQixhQUFhLENBQzNCLE9BQXNDO0lBRXRDLElBQUksT0FBTyxZQUFZLFVBQVU7UUFBRSxPQUFPLE9BQU8sQ0FBQTtJQUNqRCxJQUFJLHlCQUFpQixJQUFJLE9BQU8sWUFBWSx5QkFBaUIsQ0FBQyxTQUFTO1FBQ3JFLE9BQU8sT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFBO0lBRXpCLE1BQU0sSUFBSSxLQUFLLENBQUMsMEJBQTBCLENBQUMsQ0FBQTtBQUM3QyxDQUFDO0FBUkQsc0NBUUM7QUFFRCxTQUFnQiw0QkFBNEIsQ0FDMUMsT0FBc0M7SUFFdEMsSUFBSSx5QkFBaUIsRUFBRTtRQUNyQixJQUFJLE9BQU8sWUFBWSxVQUFVLEVBQUU7WUFDakMsTUFBTSxFQUFFLEdBQUcseUJBQWlCLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxDQUFBO1lBQ3JELHNFQUFzRTtZQUN0RSxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFBO1lBQ2YsT0FBTyxFQUFFLENBQUE7U0FDVjtRQUNELElBQUksT0FBTyxZQUFZLHlCQUFpQixDQUFDLFNBQVM7WUFBRSxPQUFPLE9BQU8sQ0FBQTtLQUNuRTtTQUFNLElBQUksT0FBTyxZQUFZLFVBQVUsRUFBRTtRQUN4QyxPQUFPLE9BQU8sQ0FBQTtLQUNmO0lBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxDQUFBO0FBQzdDLENBQUM7QUFmRCxvRUFlQyJ9